// Authentication JavaScript

// Handle Login
async function handleLogin(event) {
    event.preventDefault();
    
    const form = event.target;
    const formData = new FormData(form);
    const data = Object.fromEntries(formData);
    
    // Client-side validation
    if (!data.phone || !data.password) {
        showAlert('ফোন নাম্বার এবং পাসওয়ার্ড প্রয়োজন।', 'ত্রুটি', 'error');
        return false;
    }
    
    // Validate phone number format
    const phoneRegex = /^01[3-9]\d{8}$/;
    if (!phoneRegex.test(data.phone)) {
        showAlert('সঠিক ফোন নাম্বার দিন।', 'ত্রুটি', 'error');
        return false;
    }
    
    try {
        const email = `${data.phone}@botclub.com`;
        await auth.signInWithEmailAndPassword(email, data.password);
        
        // Send login success to server for logging
        await fetch('api/log_login.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                phone: data.phone,
                timestamp: Date.now(),
                csrf_token: CSRF_TOKEN
            })
        });
        
    } catch (error) {
        handleError(error, 'login');
        return false;
    }
    
    return false;
}

// Handle Registration
async function handleRegister(event) {
    event.preventDefault();
    
    const form = event.target;
    const formData = new FormData(form);
    const data = Object.fromEntries(formData);
    
    // Validation
    if (data.password !== data.confirm_password) {
        showAlert('পাসওয়ার্ড মেলেনি।', 'ত্রুটি', 'error');
        return false;
    }
    
    if (data.password.length < 6) {
        showAlert('পাসওয়ার্ড কমপক্ষে ৬ অক্ষরের হতে হবে।', 'ত্রুটি', 'error');
        return false;
    }
    
    const phoneRegex = /^01[3-9]\d{8}$/;
    if (!phoneRegex.test(data.phone)) {
        showAlert('সঠিক ফোন নাম্বার দিন।', 'ত্রুটি', 'error');
        return false;
    }
    
    try {
        const email = `${data.phone}@botclub.com`;
        const userCredential = await auth.createUserWithEmailAndPassword(email, data.password);
        
        // Create user profile in database
        await database.ref('users/' + userCredential.user.uid).set({
            uid: userCredential.user.uid,
            uiId: Math.floor(100000 + Math.random() * 900000).toString(),
            name: data.name,
            phone: data.phone,
            password: data.password, // Note: In production, hash this password
            depositBalance: 0,
            winningBalance: 0,
            totalReferrals: 0,
            totalBonus: 0,
            usedReferralCode: data.referral_code || null,
            createdAt: firebase.database.ServerValue.TIMESTAMP,
            hasDeposited: false,
            status: 'active'
        });
        
        showAlert('রেজিস্ট্রেশন সফল হয়েছে!', 'স্বাগতম');
        
    } catch (error) {
        handleError(error, 'registration');
        return false;
    }
    
    return false;
}

// Logout
function handleLogout() {
    if (confirm('আপনি কি লগআউট করতে চান?')) {
        auth.signOut().then(() => {
            window.location.href = 'index.php';
        });
    }
}

// Update auth forms visibility
document.querySelectorAll('.auth-tab').forEach(tab => {
    tab.addEventListener('click', () => {
        document.querySelectorAll('.auth-tab').forEach(t => t.classList.remove('active'));
        tab.classList.add('active');
        
        document.getElementById('login-form').style.display = 
            tab.dataset.auth === 'login' ? 'flex' : 'none';
        document.getElementById('register-form').style.display = 
            tab.dataset.auth === 'register' ? 'flex' : 'none';
    });
});

// Auto-focus on form inputs
document.getElementById('login-phone')?.focus();